import { ValueType } from "@opentelemetry/api";
import { hrTimeDuration, hrTimeToMilliseconds } from "@opentelemetry/core";
import { GENKIT_VERSION, GenkitError } from "genkit";
import { logger } from "genkit/logging";
import { toDisplayPath } from "genkit/tracing";
import {
  MetricCounter,
  MetricHistogram,
  internalMetricNamespaceWrap
} from "../metrics.js";
import {
  createCommonLogAttributes,
  extractErrorName,
  truncate,
  truncatePath
} from "../utils.js";
class FeaturesTelemetry {
  /**
   * Wraps the declared metrics in a Genkit-specific, internal namespace.
   */
  _N = internalMetricNamespaceWrap.bind(null, "feature");
  featureCounter = new MetricCounter(this._N("requests"), {
    description: "Counts calls to genkit features.",
    valueType: ValueType.INT
  });
  featureLatencies = new MetricHistogram(this._N("latency"), {
    description: "Latencies when calling Genkit features.",
    valueType: ValueType.DOUBLE,
    unit: "ms"
  });
  tick(span, logInputAndOutput, projectId) {
    const attributes = span.attributes;
    const name = attributes["genkit:name"];
    const path = attributes["genkit:path"];
    const latencyMs = hrTimeToMilliseconds(
      hrTimeDuration(span.startTime, span.endTime)
    );
    const isRoot = attributes["genkit:isRoot"];
    if (!isRoot) {
      throw new GenkitError({
        status: "FAILED_PRECONDITION",
        message: "FeatureTelemetry tick called with non-root span."
      });
    }
    const state = attributes["genkit:state"];
    if (state === "success") {
      this.writeFeatureSuccess(name, latencyMs);
    } else if (state === "error") {
      const errorName = extractErrorName(span.events) || "<unknown>";
      this.writeFeatureFailure(name, latencyMs, errorName);
    } else {
      logger.warn(`Unknown state; ${state}`);
      return;
    }
    if (logInputAndOutput) {
      const input = truncate(attributes["genkit:input"]);
      const output = truncate(attributes["genkit:output"]);
      const sessionId = attributes["genkit:sessionId"];
      const threadName = attributes["genkit:threadName"];
      if (input) {
        this.writeLog(
          span,
          "Input",
          name,
          path,
          input,
          projectId,
          sessionId,
          threadName
        );
      }
      if (output) {
        this.writeLog(
          span,
          "Output",
          name,
          path,
          output,
          projectId,
          sessionId,
          threadName
        );
      }
    }
  }
  writeFeatureSuccess(featureName, latencyMs) {
    const dimensions = {
      name: featureName,
      status: "success",
      source: "ts",
      sourceVersion: GENKIT_VERSION
    };
    this.featureCounter.add(1, dimensions);
    this.featureLatencies.record(latencyMs, dimensions);
  }
  writeFeatureFailure(featureName, latencyMs, errorName) {
    const dimensions = {
      name: featureName,
      status: "failure",
      source: "ts",
      sourceVersion: GENKIT_VERSION,
      error: errorName
    };
    this.featureCounter.add(1, dimensions);
    this.featureLatencies.record(latencyMs, dimensions);
  }
  writeLog(span, tag, featureName, qualifiedPath, content, projectId, sessionId, threadName) {
    const path = truncatePath(toDisplayPath(qualifiedPath));
    const sharedMetadata = {
      ...createCommonLogAttributes(span, projectId),
      path,
      qualifiedPath,
      featureName,
      sessionId,
      threadName
    };
    logger.logStructured(`${tag}[${path}, ${featureName}]`, {
      ...sharedMetadata,
      content
    });
  }
}
const featuresTelemetry = new FeaturesTelemetry();
export {
  featuresTelemetry
};
//# sourceMappingURL=feature.mjs.map