import * as trpcExpress from '@trpc/server/adapters/express';
import * as bodyParser from 'body-parser';
import * as clc from 'colorette';
import express from 'express';
import os from 'os';
import path from 'path';
import { writeToolsInfoFile } from '../utils';
import { logger } from '../utils/logger';
import { toolsPackage } from '../utils/package';
import { downloadAndExtractUiAssets } from '../utils/ui-assets';
import { TOOLS_SERVER_ROUTER } from './router';
const MAX_PAYLOAD_SIZE = 30000000;
const UI_ASSETS_GCS_BUCKET = `https://storage.googleapis.com/genkit-assets`;
const UI_ASSETS_ZIP_FILE_NAME = `${toolsPackage.version}.zip`;
const UI_ASSETS_ZIP_GCS_PATH = `${UI_ASSETS_GCS_BUCKET}/${UI_ASSETS_ZIP_FILE_NAME}`;
const UI_ASSETS_ROOT = path.resolve(os.homedir(), '.genkit', 'assets', toolsPackage.version);
const UI_ASSETS_SERVE_PATH = path.resolve(UI_ASSETS_ROOT, 'ui', 'browser');
const API_BASE_PATH = '/api';
export function startServer(manager, port) {
    let server;
    const app = express();
    downloadAndExtractUiAssets({
        fileUrl: UI_ASSETS_ZIP_GCS_PATH,
        extractPath: UI_ASSETS_ROOT,
        zipFileName: UI_ASSETS_ZIP_FILE_NAME,
    });
    app.use(express.static(UI_ASSETS_SERVE_PATH));
    app.options('/api/streamAction', async (req, res) => {
        res.setHeader('Access-Control-Allow-Origin', '*');
        res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
        res.status(200).send('');
    });
    app.post('/api/streamAction', bodyParser.json({ limit: MAX_PAYLOAD_SIZE }), async (req, res) => {
        const { key, input, context } = req.body;
        res.writeHead(200, {
            'Access-Control-Allow-Origin': '*',
            'Access-Control-Allow-Headers': 'Content-Type',
            'Content-Type': 'text/plain',
            'Transfer-Encoding': 'chunked',
        });
        try {
            const result = await manager.runAction({ key, input, context }, (chunk) => {
                res.write(JSON.stringify(chunk) + '\n');
            });
            res.write(JSON.stringify(result));
        }
        catch (err) {
            res.write(JSON.stringify({ error: err.data }));
        }
        res.end();
    });
    app.get('/api/__health', (_, res) => {
        res.status(200).send('');
    });
    app.post('/api/__quitquitquit', (_, res) => {
        logger.debug('Shutting down tools API');
        res.status(200).send('Server is shutting down');
        server.close(() => {
            process.exit(0);
        });
    });
    app.use(API_BASE_PATH, (req, res, next) => {
        res.setHeader('Access-Control-Allow-Origin', '*');
        res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
        if (req.method === 'OPTIONS')
            res.send('');
        else
            next();
    }, trpcExpress.createExpressMiddleware({
        router: TOOLS_SERVER_ROUTER(manager),
        maxBodySize: MAX_PAYLOAD_SIZE,
    }));
    app.all('*', (_, res) => {
        res.status(200).sendFile('/', { root: UI_ASSETS_SERVE_PATH });
    });
    const errorHandler = (error, request, response, next) => {
        if (error instanceof Error) {
            logger.error(error.stack);
        }
        return response.status(500).send(error);
    };
    app.use(errorHandler);
    server = app.listen(port, async () => {
        const uiUrl = 'http://localhost:' + port;
        const projectRoot = manager.projectRoot;
        logger.info(`${clc.green(clc.bold('Project root:'))} ${projectRoot}`);
        logger.info(`${clc.green(clc.bold('Genkit Developer UI:'))} ${uiUrl}`);
        await writeToolsInfoFile(uiUrl, projectRoot);
    });
    return new Promise((resolve) => {
        server.once('close', resolve);
    });
}
//# sourceMappingURL=server.js.map