"use strict";
// Copyright 2017 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.LOGGING_SAMPLED_KEY = exports.LOGGING_SPAN_KEY = exports.LOGGING_TRACE_KEY = exports.LoggingWinston = exports.getCurrentTraceFromAgent = exports.getDefaultMetadataForTracing = exports.express = void 0;
const TransportStream = require("winston-transport");
const common_1 = require("./common");
Object.defineProperty(exports, "getCurrentTraceFromAgent", { enumerable: true, get: function () { return common_1.getCurrentTraceFromAgent; } });
const express = require("./middleware/express");
exports.express = express;
const default_metadata_1 = require("./default-metadata");
Object.defineProperty(exports, "getDefaultMetadataForTracing", { enumerable: true, get: function () { return default_metadata_1.getDefaultMetadataForTracing; } });
const LEVEL = Symbol.for('level');
/**
 * This module provides support for streaming your winston logs to
 * [Cloud Logging](https://cloud.google.com/logging).
 *
 * @class
 *
 * @param {object} [options]
 * @param {object} [options.level] The default log level. Winston will filter
 *     messages with a severity lower than this.
 * @param {object} [options.levels] Custom logging levels as supported by
 *     winston. This list is used to translate your log level to the Cloud
 *     Logging level. Each property should have an integer value between 0 (most
 *     severe) and 7 (least severe). If you are passing a list of levels to your
 *     winston logger, you should provide the same list here.
 * @param {boolean} [options.inspectMetadata=false] Serialize winston-provided log
 *     metadata using `util.inspect`.
 * @param {string} [options.logName=winston_log] The name of the log that will receive
 *     messages written to this transport.
 * @param {object} [options.resource] The monitored resource that the transport
 *     corresponds to. On Google Cloud Platform, this is detected automatically,
 *     but you may optionally specify a specific monitored resource. For more
 *     information see the
 *     [official documentation]{@link
 * https://cloud.google.com/logging/docs/api/reference/rest/v2/MonitoredResource}.
 * @param {object} [options.serviceContext] For logged errors, we provide this
 *     as the service context. For more information see
 *     [this guide]{@link
 * https://cloud.google.com/error-reporting/docs/formatting-error-messages} and
 * the [official documentation]{@link
 * https://cloud.google.com/error-reporting/reference/rest/v1beta1/ServiceContext}.
 * @param {string} [options.serviceContext.service] An identifier of the
 *     service, such as the name of the executable, job, or Google App Engine
 *     service name.
 * @param {string} [options.serviceContext.version] Represents the version of
 *     the service.
 * @param {string} [options.projectId] The project ID from the Google Cloud
 *     Console, e.g. 'grape-spaceship-123'. We will also check the environment
 *     variable `GCLOUD_PROJECT` for your project ID. If your app is running in
 *     an environment which supports {@link
 * https://cloud.google.com/docs/authentication/production#providing_credentials_to_your_application
 * Application Default Credentials}, your project ID will be detected
 * automatically.
 * @param {string} [options.keyFilename] Full path to the a .json, .pem, or .p12
 *     key downloaded from the Google Cloud Console. If you provide a path
 *     to a JSON file, the `projectId` option above is not necessary. NOTE: .pem
 *     and .p12 require you to specify the `email` option as well.
 * @param {string} [options.email] Account email address. Required when using a
 *     .pem or .p12 keyFilename.
 * @param {object} [options.credentials] Credentials object.
 * @param {string} [options.credentials.client_email]
 * @param {string} [options.credentials.private_key]
 * @param {boolean} [options.autoRetry=true] Automatically retry requests if the
 *     response is related to rate limits or certain intermittent server errors.
 *     We will exponentially backoff subsequent requests by default.
 * @param {number} [options.maxRetries=3] Maximum number of automatic retries
 *     attempted before returning the error.
 * @param {constructor} [options.promise] Custom promise module to use instead
 *     of native Promises.
 *
 * @example <caption>Import the client library</caption>
 * const {LoggingWinston} = require('@google-cloud/logging-winston');
 *
 * @example <caption>Create a client that uses <a
 * href="https://cloud.google.com/docs/authentication/production#providing_credentials_to_your_application">Application
 * Default Credentials (ADC)</a>:</caption> const loggingWinston = new
 * LoggingWinston();
 *
 * @example <caption>Create a client with <a
 * href="https://cloud.google.com/docs/authentication/production#obtaining_and_providing_service_account_credentials_manually">explicit
 * credentials</a>:</caption> const loggingWinston = new LoggingWinston({
 *   projectId: 'your-project-id',
 *   keyFilename: '/path/to/keyfile.json'
 * });
 *
 * @example <caption>include:samples/quickstart.js</caption>
 * region_tag:logging_winston_quickstart
 * Full quickstart example:
 */
class LoggingWinston extends TransportStream {
    constructor(options) {
        options = options || {};
        super({
            level: options.level,
            format: options.format,
            silent: options.silent,
            handleExceptions: options.handleExceptions,
            handleRejections: options.handleRejections,
        });
        this.common = new common_1.LoggingCommon(options);
    }
    // eslint-disable-next-line
    log(info, callback) {
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        const { message, level, splat, stack, ...metadata } = info;
        // If the whole message is an error we have to manually copy the stack into
        // metadata. Errors dont have enumerable properties so they don't
        // destructure.
        if (stack)
            metadata.stack = stack;
        this.common.log(info[LEVEL] || level, message, metadata || {}, callback);
    }
}
exports.LoggingWinston = LoggingWinston;
LoggingWinston.LOGGING_TRACE_KEY = common_1.LOGGING_TRACE_KEY;
LoggingWinston.LOGGING_SPAN_KEY = common_1.LOGGING_SPAN_KEY;
LoggingWinston.LOGGING_SAMPLED_KEY = common_1.LOGGING_SAMPLED_KEY;
// LOGGING_TRACE_KEY is Cloud Logging specific and has the format:
// logging.googleapis.com/trace
// For more information, see: https://cloud.google.com/logging/docs/reference/v2/rpc/google.logging.v2#logentry
exports.LOGGING_TRACE_KEY = common_1.LOGGING_TRACE_KEY;
// LOGGING_SPAN_KEY is Cloud Logging specific and has the format:
// logging.googleapis.com/spanId
// For more information, see: https://cloud.google.com/logging/docs/reference/v2/rpc/google.logging.v2#logentry
exports.LOGGING_SPAN_KEY = common_1.LOGGING_SPAN_KEY;
// LOGGING_SAMPLED_KEY is Cloud Logging specific and has the format:
// logging.googleapis.com/trace_sampled
// The value of this field must be either true or false. For more information,
// see traceSampled on the LogEntry page:	https://cloud.google.com/logging/docs/reference/v2/rpc/google.logging.v2#logentry
exports.LOGGING_SAMPLED_KEY = common_1.LOGGING_SAMPLED_KEY;
//# sourceMappingURL=index.js.map