import {
  modelActionMetadata,
  z
} from "genkit";
import {
  getBasicUsageStats,
  modelRef
} from "genkit/model";
import { imagenPredict } from "./client.js";
import {
  calculateApiKey,
  checkApiKey,
  checkModelName,
  extractImagenImage,
  extractText,
  extractVersion,
  modelName
} from "./utils.js";
const ImagenConfigSchema = z.object({
  apiKey: z.string().describe("Override the API key provided at plugin initialization.").optional(),
  numberOfImages: z.number().describe(
    "The number of images to generate, from 1 to 4 (inclusive). The default is 1."
  ).optional(),
  aspectRatio: z.enum(["1:1", "9:16", "16:9", "3:4", "4:3"]).describe("Desired aspect ratio of the output image.").optional(),
  personGeneration: z.enum(["dont_allow", "allow_adult", "allow_all"]).describe(
    "Control if/how images of people will be generated by the model."
  ).optional()
}).passthrough();
function commonRef(name, info, configSchema = ImagenConfigSchema) {
  return modelRef({
    name: `googleai/${name}`,
    configSchema,
    info: info ?? {
      supports: {
        media: true,
        multiturn: false,
        tools: false,
        toolChoice: false,
        systemRole: false,
        output: ["media"]
      }
    }
  });
}
const GENERIC_MODEL = commonRef("imagen", {
  supports: {
    media: true,
    multiturn: true,
    tools: true,
    systemRole: true,
    output: ["media"]
  }
});
const KNOWN_MODELS = {
  "imagen-3.0-generate-002": commonRef("imagen-3.0-generate-002"),
  "imagen-4.0-generate-preview-06-06": commonRef(
    "imagen-4.0-generate-preview-06-06"
  ),
  "imagen-4.0-ultra-generate-preview-06-06": commonRef(
    "imagen-4.0-ultra-generate-preview-06-06"
  )
};
function isImagenModelName(value) {
  return !!value?.startsWith("imagen-");
}
function model(version, config = {}) {
  const name = checkModelName(version);
  if (KNOWN_MODELS[name]) {
    return KNOWN_MODELS[name].withConfig(config);
  }
  return modelRef({
    name: `googleai/${name}`,
    config,
    configSchema: ImagenConfigSchema,
    info: {
      ...GENERIC_MODEL.info
    }
  });
}
function listActions(models) {
  return models.filter(
    (m) => m.supportedGenerationMethods.includes("predict") && isImagenModelName(modelName(m.name))
  ).filter((m) => !m.description || !m.description.includes("deprecated")).map((m) => {
    const ref = model(m.name);
    return modelActionMetadata({
      name: ref.name,
      info: ref.info,
      configSchema: ref.configSchema
    });
  });
}
function defineKnownModels(ai, options) {
  for (const name of Object.keys(KNOWN_MODELS)) {
    defineModel(ai, name, options);
  }
}
function defineModel(ai, name, pluginOptions) {
  checkApiKey(pluginOptions?.apiKey);
  const ref = model(name);
  const clientOptions = {
    apiVersion: pluginOptions?.apiVersion,
    baseUrl: pluginOptions?.baseUrl
  };
  return ai.defineModel(
    {
      apiVersion: "v2",
      name: ref.name,
      ...ref.info,
      configSchema: ref.configSchema
    },
    async (request, { abortSignal }) => {
      const clientOpt = { ...clientOptions, signal: abortSignal };
      const imagenPredictRequest = {
        instances: [
          {
            prompt: extractText(request),
            image: extractImagenImage(request)
          }
        ],
        parameters: toImagenParameters(request)
      };
      const predictApiKey = calculateApiKey(
        pluginOptions?.apiKey,
        request.config?.apiKey
      );
      const response = await imagenPredict(
        predictApiKey,
        extractVersion(ref),
        imagenPredictRequest,
        clientOpt
      );
      if (!response.predictions || response.predictions.length == 0) {
        throw new Error(
          "Model returned no predictions. Possibly due to content filters."
        );
      }
      const message = {
        role: "model",
        content: response.predictions.map(fromImagenPrediction)
      };
      return {
        finishReason: "stop",
        message,
        usage: getBasicUsageStats(request.messages, message),
        custom: response
      };
    }
  );
}
function fromImagenPrediction(p) {
  const b64data = p.bytesBase64Encoded;
  const mimeType = p.mimeType;
  return {
    media: {
      url: `data:${mimeType};base64,${b64data}`,
      contentType: mimeType
    }
  };
}
function toImagenParameters(request) {
  const out = {
    sampleCount: request.config?.numberOfImages ?? 1,
    ...request?.config
  };
  for (const k in out) {
    if (!out[k]) delete out[k];
  }
  delete out.apiKey;
  return out;
}
const TEST_ONLY = {
  toImagenParameters,
  fromImagenPrediction,
  GENERIC_MODEL,
  KNOWN_MODELS
};
export {
  ImagenConfigSchema,
  TEST_ONLY,
  defineKnownModels,
  defineModel,
  isImagenModelName,
  listActions,
  model
};
//# sourceMappingURL=imagen.mjs.map