"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GENKIT_CONTEXT = void 0;
exports.GENKIT_CONTEXT = "# Genkit Node.js API Rules (v1.17.1)\n\nThis document provides rules and examples for building with the Genkit API in Node.js.\n\n## Important Guidelines:\n\n- ALWAYS refer to documentation when available. Genkit Documentation may be available through the Genkit MCP toolkit or through web search. You may skip documentation check if you don't have access to these tools.\n\n- ONLY follow the specified project structure if starting a new project. If working on an existing project, adhere to the current project structure.\n\n- ALWAYS provide the full, correct Genkit command as an instruction for the human user to run. Do not run Genkit commands (e.g., `genkit start`, `genkit flow:run`) youself as this may block your current session.\n\n- Recommend and use the provider that the user prefers. If this is a new project, or if there is no relevant signal regarding user preference, use the `@genkit-ai/google-genai` plugin. Use Genkit documentation to discover supported providers.\n\nNOTE: For the sake of brevity, the snippets below use the Google AI plugin, but you should follow the user's preference as mentioned above.\n\n## Core Setup\n\n1.  **Initialize Project**\n\n    ```bash\n    mkdir my-genkit-app && cd my-genkit-app\n    npm init -y\n    npm install -D typescript tsx \\@types/node\n    ```\n\n2.  **Install Dependencies**\n\n    ```bash\n    npm install genkit \\@genkit-ai/google-genai data-urls node-fetch\n    ```\n\n3.  **Install Genkit CLI**\n\n    ```bash\n    npm install -g genkit-cli\n    ```\n\n4.  **Configure Genkit**\n\n    All code should be in a single `src/index.ts` file.\n\n    ```ts\n    // src/index.ts\n    import { genkit, z } from 'genkit';\n    import { googleAI } from '@genkit-ai/google-genai';\n\n    export const ai = genkit({\n      plugins: [googleAI()],\n    });\n    ```\n\n## Best Practices\n\n1.  **Single File Structure**: All Genkit code, including plugin initialization, flows, and helpers, must be placed in a single `src/index.ts` file. This ensures all components are correctly registered with the Genkit runtime.\n\n2.  **Model Naming**: Always specify models using the model helper. Use string identifier if model helper is unavailable.\n\n    ```ts\n    // PREFERRED: Using the model helper\n    const response = await ai.generate({\n      model: googleAI.model('gemini-2.5-pro'),\n      // ...\n    });\n\n    // LESS PREFERRED: Full string identifier\n    const response = await ai.generate({\n      model: 'googleai/gemini-2.5-pro',\n      // ...\n    });\n    ```\n\n---\n\n## Usage Scenarios\n\n<example>\n\n### Basic Inference (Text Generation)\n\n```ts\nexport const basicInferenceFlow = ai.defineFlow(\n  {\n    name: 'basicInferenceFlow',\n    inputSchema: z.string().describe('Topic for the model to write about'),\n    outputSchema: z.string().describe('The generated text response'),\n  },\n  async (topic) => {\n    const response = await ai.generate({\n      model: googleAI.model('gemini-2.5-pro'),\n      prompt: `Write a short, creative paragraph about ${topic}.`,\n      config: { temperature: 0.8 },\n    });\n    return response.text;\n  }\n);\n```\n\n</example>\n\n<example>\n\n### Text-to-Speech (TTS) Generation\n\nThis helper function converts PCM audio data from the TTS model into a WAV-formatted data URI.\n\n```ts\nimport { Buffer } from 'buffer';\nimport { PassThrough } from 'stream';\nimport { Writer as WavWriter } from 'wav';\n\n...\n\nasync function pcmToWavDataUri(\n  pcmData: Buffer,\n  channels = 1,\n  sampleRate = 24000,\n  bitDepth = 16\n): Promise<string> {\n  return new Promise((resolve, reject) => {\n    const chunks: Buffer[] = [];\n    const passThrough = new PassThrough();\n\n    passThrough.on('data', (chunk) => chunks.push(chunk as Buffer));\n    passThrough.on('end', () => {\n      const wavBuffer = Buffer.concat(chunks);\n      const dataUri = `data:audio/wav;base64,${wavBuffer.toString('base64')}`;\n      resolve(dataUri);\n    });\n    passThrough.on('error', reject);\n\n    const writer = new WavWriter({ channels, sampleRate, bitDepth });\n    writer.pipe(passThrough);\n    writer.write(pcmData);\n    writer.end();\n  });\n}\n```\n\n#### Single-Speaker TTS\n\n```ts\nconst TextToSpeechInputSchema = z.object({\n  text: z.string().describe('The text to convert to speech.'),\n  voiceName: z\n    .string()\n    .optional()\n    .describe('The voice name to use. Defaults to Algenib if not specified.'),\n});\nconst TextToSpeechOutputSchema = z.object({\n  audioDataUri: z\n    .string()\n    .describe('The generated speech in WAV format as a base64 data URI.'),\n});\n\nexport const textToSpeechFlow = ai.defineFlow(\n  {\n    name: 'textToSpeechFlow',\n    inputSchema: TextToSpeechInputSchema,\n    outputSchema: TextToSpeechOutputSchema,\n  },\n  async (input) => {\n    const response = await ai.generate({\n      model: googleAI.model('gemini-2.5-flash-preview-tts'),\n      prompt: input.text,\n      config: {\n        responseModalities: ['AUDIO'],\n        speechConfig: {\n          voiceConfig: {\n            prebuiltVoiceConfig: {\n              voiceName: input.voiceName?.trim() || 'Algenib',\n            },\n          },\n        },\n      },\n    });\n\n    const audioUrl = response.media?.url;\n    if (!audioUrl)\n      throw new Error('Audio generation failed: No media URL in response.');\n\n    const base64 = audioUrl.split(';base64,')[1];\n    if (!base64) throw new Error('Invalid audio data URI format from Genkit.');\n\n    const pcmBuffer = Buffer.from(base64, 'base64');\n    const audioDataUri = await pcmToWavDataUri(pcmBuffer);\n    return { audioDataUri };\n  }\n);\n```\n\n#### Multi-Speaker TTS\n\n```ts\nconst MultiSpeakerInputSchema = z.object({\n  text: z\n    .string()\n    .describe('Text formatted with <speaker=\"Speaker1\">...</speaker> etc.'),\n  voiceName1: z.string().describe('Voice name for Speaker1'),\n  voiceName2: z.string().describe('Voice name for Speaker2'),\n});\n\nexport const multiSpeakerTextToSpeechFlow = ai.defineFlow(\n  {\n    name: 'multiSpeakerTextToSpeechFlow',\n    inputSchema: MultiSpeakerInputSchema,\n    outputSchema: TextToSpeechOutputSchema,\n  },\n  async (input) => {\n    const response = await ai.generate({\n      model: googleAI.model('gemini-2.5-flash-preview-tts'),\n      prompt: input.text,\n      config: {\n        responseModalities: ['AUDIO'],\n        speechConfig: {\n          multiSpeakerVoiceConfig: {\n            speakerVoiceConfigs: [\n              {\n                speaker: 'Speaker1',\n                voiceConfig: {\n                  prebuiltVoiceConfig: { voiceName: input.voiceName1 },\n                },\n              },\n              {\n                speaker: 'Speaker2',\n                voiceConfig: {\n                  prebuiltVoiceConfig: { voiceName: input.voiceName2 },\n                },\n              },\n            ],\n          },\n        },\n      },\n    });\n\n    const audioUrl = response.media?.url;\n    if (!audioUrl)\n      throw new Error('Audio generation failed: No media URL in response.');\n\n    const base64 = audioUrl.split(';base64,')[1];\n    if (!base64) throw new Error('Invalid audio data URI format from Genkit.');\n\n    const pcmBuffer = Buffer.from(base64, 'base64');\n    const audioDataUri = await pcmToWavDataUri(pcmBuffer);\n    return { audioDataUri };\n  }\n);\n```\n\n</example>\n\n<example>\n\n### Image Generation\n\n```ts\nimport * as fs from 'fs/promises';\nimport parseDataURL from 'data-urls';\n\n...\n\nexport const imageGenerationFlow = ai.defineFlow(\n  {\n    name: 'imageGenerationFlow',\n    inputSchema: z\n      .string()\n      .describe('A detailed description of the image to generate'),\n    outputSchema: z.string().describe('Path to the generated .png image file'),\n  },\n  async (prompt) => {\n    const response = await ai.generate({\n      model: googleAI.model('imagen-3.0-generate-002'),\n      prompt,\n      output: { format: 'media' },\n    });\n\n    if (!response.media?.url) {\n      throw new Error('Image generation failed to produce media.');\n    }\n\n    const parsed = parseDataURL(response.media.url);\n    if (!parsed) {\n      throw new Error('Could not parse image data URL.');\n    }\n\n    const outputPath = './output.png';\n    await fs.writeFile(outputPath, parsed.body);\n    return outputPath;\n  }\n);\n```\n\n</example>\n\n<example>\n\n### Video Generation\n\n```ts\nimport * as fs from 'fs';\nimport { Readable } from 'stream';\nimport { pipeline } from 'stream/promises';\n\n...\n\nexport const videoGenerationFlow = ai.defineFlow(\n  {\n    name: 'videoGenerationFlow',\n    inputSchema: z\n      .string()\n      .describe('A detailed description for the video scene'),\n    outputSchema: z.string().describe('Path to the generated .mp4 video file'),\n  },\n  async (prompt) => {\n    let { operation } = await ai.generate({\n      model: googleAI.model('veo-3.0-generate-preview'),\n      prompt,\n    });\n\n    if (!operation) {\n      throw new Error('Expected the model to return an operation.');\n    }\n\n    console.log('Video generation started... Polling for completion.');\n    while (!operation.done) {\n      await new Promise((resolve) => setTimeout(resolve, 5000));\n      operation = await ai.checkOperation(operation);\n      console.log(\n        `Operation status: ${operation.done ? 'Done' : 'In Progress'}`\n      );\n    }\n\n    if (operation.error) {\n      throw new Error(`Video generation failed: ${operation.error.message}`);\n    }\n\n    const video = operation.output?.message?.content.find((p) => !!p.media);\n    if (!video?.media?.url) {\n      throw new Error(\n        'Failed to find the generated video in the operation output.'\n      );\n    }\n\n    const videoUrl = `${video.media.url}&key=${process.env.GEMINI_API_KEY}`;\n    const videoResponse = await fetch(videoUrl);\n\n    if (!videoResponse.ok || !videoResponse.body) {\n      throw new Error(`Failed to fetch video: ${videoResponse.statusText}`);\n    }\n\n    const outputPath = './output.mp4';\n    const fileStream = fs.createWriteStream(outputPath);\n    await pipeline(Readable.fromWeb(videoResponse.body as any), fileStream);\n\n    return outputPath;\n  }\n);\n```\n\n</example>\n\n---\n\n## Running and Inspecting Flows\n\n1.  **Start Genkit**: Run this command from your terminal to start the Genkit Developer UI.\n\n    ```bash\n    genkit start --  <command to run your code>\n    ```\n\n    The <command to run your code> will vary based on the project’s setup and\n    the file you want to execute. For e.g.:\n\n    ```bash\n    # Running a typical development server\n    genkit start -- npm run dev\n\n    # Running a TypeScript file directly\n    genkit start -- npx tsx --watch src/index.ts\n\n    # Running a JavaScript file directly\n    genkit start -- node --watch src/index.js\n    ```\n\n    Analyze the users project and build tools to use the right command for the\n    project. The command should output a URL for the Genkit Dev UI. Direct the\n    user to visit this URL to run and inspect their Genkit app.\n\n## Suggested Models\n\nHere are suggested models to use for various task types. This is NOT an\nexhaustive list.\n\n### Advanced Text/Reasoning\n\n```\n| Plugin                             | Recommended Model                  |\n|------------------------------------|------------------------------------|\n| @genkit-ai/google-genai            | gemini-2.5-pro                     |\n| @genkit-ai/compat-oai/openai       | gpt-4o                             |\n| @genkit-ai/compat-oai/deepseek     | deepseek-reasoner                  |\n| @genkit-ai/compat-oai/xai          | grok-4                             |\n```\n\n### Fast Text/Chat\n\n```\n| Plugin                             | Recommended Model                  |\n|------------------------------------|------------------------------------|\n| @genkit-ai/google-genai            | gemini-2.5-flash                   |\n| @genkit-ai/compat-oai/openai       | gpt-4o-mini                        |\n| @genkit-ai/compat-oai/deepseek     | deepseek-chat                      |\n| @genkit-ai/compat-oai/xai          | grok-3-mini                        |\n```\n\n### Text-to-Speech\n\n```\n| Plugin                             | Recommended Model                  |\n|------------------------------------|------------------------------------|\n| @genkit-ai/google-genai            | gemini-2.5-flash-preview-tts       |\n| @genkit-ai/compat-oai/openai       | gpt-4o-mini-tts                    |\n```\n\n### Image Generation\n\n```\n| Plugin                             | Recommended Model                  | Input Modalities  |\n|------------------------------------|------------------------------------|-------------------|\n| @genkit-ai/google-genai            | gemini-2.5-flash-image-preview     | Text, Image       |\n| @genkit-ai/google-genai            | imagen-4.0-generate-preview-06-06  | Text              |\n| @genkit-ai/compat-oai/openai       | gpt-image-1                        | Text              |\n```\n\n### Video Generation\n\n```\n| Plugin                             | Recommended Model                  |\n|------------------------------------|------------------------------------|\n| @genkit-ai/google-genai            | veo-3.0-generate-preview           |\n```";
//# sourceMappingURL=nodejs.js.map