import { ValueType } from "@opentelemetry/api";
import { hrTimeDuration, hrTimeToMilliseconds } from "@opentelemetry/core";
import { GENKIT_VERSION } from "genkit";
import { logger } from "genkit/logging";
import { toDisplayPath } from "genkit/tracing";
import {
  MetricCounter,
  MetricHistogram,
  internalMetricNamespaceWrap
} from "../metrics.js";
import {
  createCommonLogAttributes,
  extractErrorMessage,
  extractErrorName,
  extractErrorStack,
  extractOuterFeatureNameFromPath,
  truncatePath
} from "../utils.js";
class PathsTelemetry {
  /**
   * Wraps the declared metrics in a Genkit-specific, internal namespace.
   */
  _N = internalMetricNamespaceWrap.bind(null, "feature");
  pathCounter = new MetricCounter(this._N("path/requests"), {
    description: "Tracks unique flow paths per flow.",
    valueType: ValueType.INT
  });
  pathLatencies = new MetricHistogram(this._N("path/latency"), {
    description: "Latencies per flow path.",
    ValueType: ValueType.DOUBLE,
    unit: "ms"
  });
  tick(span, logInputAndOutput, projectId) {
    const attributes = span.attributes;
    const path = attributes["genkit:path"];
    const isFailureSource = !!span.attributes["genkit:isFailureSource"];
    const state = attributes["genkit:state"];
    if (!path || !isFailureSource || state !== "error") {
      return;
    }
    const sessionId = attributes["genkit:sessionId"];
    const threadName = attributes["genkit:threadName"];
    const errorName = extractErrorName(span.events) || "<unknown>";
    const errorMessage = extractErrorMessage(span.events) || "<unknown>";
    const errorStack = extractErrorStack(span.events) || "";
    const latency = hrTimeToMilliseconds(
      hrTimeDuration(span.startTime, span.endTime)
    );
    const pathDimensions = {
      featureName: extractOuterFeatureNameFromPath(path),
      status: "failure",
      error: errorName,
      path,
      source: "ts",
      sourceVersion: GENKIT_VERSION
    };
    this.pathCounter.add(1, pathDimensions);
    this.pathLatencies.record(latency, pathDimensions);
    const displayPath = truncatePath(toDisplayPath(path));
    logger.logStructuredError(`Error[${displayPath}, ${errorName}]`, {
      ...createCommonLogAttributes(span, projectId),
      path: displayPath,
      qualifiedPath: path,
      name: errorName,
      message: errorMessage,
      stack: errorStack,
      source: "ts",
      sourceVersion: GENKIT_VERSION,
      sessionId,
      threadName
    });
  }
}
const pathsTelemetry = new PathsTelemetry();
export {
  pathsTelemetry
};
//# sourceMappingURL=path.mjs.map