"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.runNewEvaluation = runNewEvaluation;
exports.runInference = runInference;
exports.runEvaluation = runEvaluation;
exports.getAllEvaluatorActions = getAllEvaluatorActions;
exports.getMatchingEvaluatorActions = getMatchingEvaluatorActions;
const crypto_1 = require("crypto");
const _1 = require(".");
const types_1 = require("../types");
const utils_1 = require("../utils");
const parser_1 = require("./parser");
const SUPPORTED_ACTION_TYPES = ['flow', 'model'];
async function runNewEvaluation(manager, request) {
    const { dataSource, actionRef, evaluators } = request;
    const { datasetId, data } = dataSource;
    if (!datasetId && !data) {
        throw new Error(`Either 'data' or 'datasetId' must be provided`);
    }
    const hasTargetAction = await (0, utils_1.hasAction)({ manager, actionRef });
    if (!hasTargetAction) {
        throw new Error(`Cannot find action ${actionRef}.`);
    }
    let inferenceDataset;
    let metadata = {};
    if (datasetId) {
        const datasetStore = await (0, _1.getDatasetStore)();
        utils_1.logger.info(`Fetching dataset ${datasetId}...`);
        const dataset = await datasetStore.getDataset(datasetId);
        if (dataset.length === 0) {
            throw new Error(`Dataset ${datasetId} is empty`);
        }
        inferenceDataset = types_1.DatasetSchema.parse(dataset);
        const datasetMetadatas = await datasetStore.listDatasets();
        const targetDatasetMetadata = datasetMetadatas.find((d) => d.datasetId === datasetId);
        const datasetVersion = targetDatasetMetadata?.version;
        metadata = { datasetId, datasetVersion };
    }
    else {
        const rawData = data.map((sample) => ({
            ...sample,
            testCaseId: sample.testCaseId ?? (0, utils_1.generateTestCaseId)(),
        }));
        inferenceDataset = types_1.DatasetSchema.parse(rawData);
    }
    utils_1.logger.info('Running inference...');
    const evalDataset = await runInference({
        manager,
        actionRef,
        inferenceDataset,
        context: request.options?.context,
        actionConfig: request.options?.actionConfig,
    });
    const evaluatorActions = await getMatchingEvaluatorActions(manager, evaluators);
    const evalRun = await runEvaluation({
        manager,
        evaluatorActions,
        evalDataset,
        batchSize: request.options?.batchSize,
        augments: {
            ...metadata,
            actionRef,
            actionConfig: request.options?.actionConfig,
        },
    });
    return evalRun.key;
}
async function runInference(params) {
    const { manager, actionRef, inferenceDataset, context, actionConfig } = params;
    if (!isSupportedActionRef(actionRef)) {
        throw new Error('Inference is only supported on flows and models');
    }
    const evalDataset = await bulkRunAction({
        manager,
        actionRef,
        inferenceDataset,
        context,
        actionConfig,
    });
    return evalDataset;
}
async function runEvaluation(params) {
    const { manager, evaluatorActions, evalDataset, augments, batchSize } = params;
    if (evalDataset.length === 0) {
        throw new Error('Cannot run evaluation, no data provided');
    }
    const evalRunId = (0, crypto_1.randomUUID)();
    const scores = {};
    utils_1.logger.info('Running evaluation...');
    const runtime = manager.getMostRecentRuntime();
    const isNodeRuntime = runtime?.genkitVersion?.startsWith('nodejs') ?? false;
    for (const action of evaluatorActions) {
        const name = (0, utils_1.evaluatorName)(action);
        const response = await manager.runAction({
            key: name,
            input: {
                dataset: evalDataset.filter((row) => !row.error),
                evalRunId,
                batchSize: isNodeRuntime ? batchSize : undefined,
            },
        });
        scores[name] = response.result;
    }
    const scoredResults = (0, parser_1.enrichResultsWithScoring)(scores, evalDataset);
    const metadata = (0, parser_1.extractMetricsMetadata)(evaluatorActions);
    const metricSummaries = (0, parser_1.extractMetricSummaries)(scores);
    const evalRun = {
        key: {
            evalRunId,
            createdAt: new Date().toISOString(),
            metricSummaries,
            ...augments,
        },
        results: scoredResults,
        metricsMetadata: metadata,
    };
    utils_1.logger.info('Finished evaluation, writing key...');
    const evalStore = await (0, _1.getEvalStore)();
    await evalStore.save(evalRun);
    return evalRun;
}
async function getAllEvaluatorActions(manager) {
    const allActions = await manager.listActions();
    const allEvaluatorActions = [];
    for (const key in allActions) {
        if ((0, utils_1.isEvaluator)(key)) {
            allEvaluatorActions.push(allActions[key]);
        }
    }
    return allEvaluatorActions;
}
async function getMatchingEvaluatorActions(manager, evaluators) {
    if (!evaluators) {
        return [];
    }
    const allEvaluatorActions = await getAllEvaluatorActions(manager);
    const filteredEvaluatorActions = allEvaluatorActions.filter((action) => evaluators.includes(action.key));
    if (filteredEvaluatorActions.length === 0) {
        if (allEvaluatorActions.length == 0) {
            throw new Error('No evaluators installed');
        }
    }
    return filteredEvaluatorActions;
}
async function bulkRunAction(params) {
    const { manager, actionRef, inferenceDataset, context, actionConfig } = params;
    const isModelAction = actionRef.startsWith('/model');
    if (inferenceDataset.length === 0) {
        throw new Error('Cannot run inference, no data provided');
    }
    const fullInferenceDataset = inferenceDataset;
    const states = [];
    const evalInputs = [];
    for (const sample of fullInferenceDataset) {
        utils_1.logger.info(`Running inference '${actionRef}' ...`);
        if (isModelAction) {
            states.push(await runModelAction({
                manager,
                actionRef,
                sample,
                modelConfig: actionConfig,
            }));
        }
        else {
            states.push(await runFlowAction({
                manager,
                actionRef,
                sample,
                context,
            }));
        }
    }
    utils_1.logger.info(`Gathering evalInputs...`);
    for (const state of states) {
        evalInputs.push(await gatherEvalInput({ manager, actionRef, state }));
    }
    return evalInputs;
}
async function runFlowAction(params) {
    const { manager, actionRef, sample, context } = { ...params };
    let state;
    try {
        const runActionResponse = await manager.runAction({
            key: actionRef,
            input: sample.input,
            context: context ? JSON.parse(context) : undefined,
        });
        state = {
            ...sample,
            traceId: runActionResponse.telemetry?.traceId,
            response: runActionResponse.result,
        };
    }
    catch (e) {
        const traceId = e?.data?.details?.traceId;
        state = {
            ...sample,
            traceId,
            evalError: `Error when running inference. Details: ${e?.message ?? e}`,
        };
    }
    return state;
}
async function runModelAction(params) {
    const { manager, actionRef, modelConfig, sample } = { ...params };
    let state;
    try {
        const modelInput = (0, utils_1.getModelInput)(sample.input, modelConfig);
        const runActionResponse = await manager.runAction({
            key: actionRef,
            input: modelInput,
        });
        state = {
            ...sample,
            traceId: runActionResponse.telemetry?.traceId,
            response: runActionResponse.result,
        };
    }
    catch (e) {
        const traceId = e?.data?.details?.traceId;
        state = {
            ...sample,
            traceId,
            evalError: `Error when running inference. Details: ${e?.message ?? e}`,
        };
    }
    return state;
}
async function gatherEvalInput(params) {
    const { manager, actionRef, state } = params;
    const extractors = await (0, utils_1.getEvalExtractors)(actionRef);
    const traceId = state.traceId;
    if (!traceId) {
        utils_1.logger.warn('No traceId available...');
        return {
            ...state,
            error: state.evalError,
            testCaseId: state.testCaseId,
            traceIds: [],
        };
    }
    const trace = await manager.getTrace({
        traceId,
    });
    const isModelAction = actionRef.startsWith('/model');
    const input = isModelAction ? state.input : extractors.input(trace);
    const nestedSpan = (0, utils_1.stackTraceSpans)(trace);
    if (!nestedSpan) {
        return {
            testCaseId: state.testCaseId,
            input,
            error: `Unable to extract any spans from trace ${traceId}`,
            reference: state.reference,
            traceIds: [traceId],
        };
    }
    if (nestedSpan.attributes['genkit:state'] === 'error') {
        return {
            testCaseId: state.testCaseId,
            input,
            error: getSpanErrorMessage(nestedSpan) ?? `Unknown error in trace ${traceId}`,
            reference: state.reference,
            traceIds: [traceId],
        };
    }
    const output = extractors.output(trace);
    const context = extractors.context(trace);
    const error = isModelAction ? getErrorFromModelResponse(output) : undefined;
    return {
        testCaseId: state.testCaseId,
        input,
        output,
        error,
        context: Array.isArray(context) ? context : [context],
        reference: state.reference,
        traceIds: [traceId],
    };
}
function getSpanErrorMessage(span) {
    if (span && span.status?.code === 2) {
        const event = span.timeEvents?.timeEvent
            ?.filter((e) => e.annotation.description === 'exception')
            .shift();
        return (event?.annotation?.attributes['exception.message'] ?? 'Error');
    }
}
function getErrorFromModelResponse(obj) {
    const response = types_1.GenerateResponseSchema.parse(obj);
    const hasLegacyResponse = !!response.candidates && response.candidates.length > 0;
    const hasNewResponse = !!response.message;
    if (!response || (!hasLegacyResponse && !hasNewResponse)) {
        return `No response was extracted from the output. '${JSON.stringify(obj)}'`;
    }
    if (hasLegacyResponse) {
        const candidate = response.candidates[0];
        if (candidate.finishReason === 'blocked') {
            return candidate.finishMessage || `Generation was blocked by the model.`;
        }
    }
    if (hasNewResponse) {
        if (response.finishReason === 'blocked') {
            return response.finishMessage || `Generation was blocked by the model.`;
        }
    }
}
function isSupportedActionRef(actionRef) {
    return SUPPORTED_ACTION_TYPES.some((supportedType) => actionRef.startsWith(`/${supportedType}`));
}
//# sourceMappingURL=evaluate.js.map