"use strict";
// Copyright 2018 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.REQUEST_LOG_SUFFIX = void 0;
exports.makeMiddleware = makeMiddleware;
const logging_1 = require("@google-cloud/logging");
const google_auth_library_1 = require("google-auth-library");
const common_1 = require("../common");
const index_1 = require("../index");
const make_child_logger_1 = require("./make-child-logger");
exports.REQUEST_LOG_SUFFIX = '_reqlog';
async function makeMiddleware(logger, optionsOrTransport, skipParentEntryForCloudRun) {
    let transport;
    // If no custom transports are provided, use default or instantiate one.
    const cloudTransport = logger.transports.find(t => t instanceof index_1.LoggingWinston);
    // If user provides a custom transport, always add it to the logger.
    if (optionsOrTransport instanceof index_1.LoggingWinston) {
        transport = optionsOrTransport;
        logger.add(transport);
    }
    else if (cloudTransport && !optionsOrTransport) {
        // Check if logger already contains a Cloud transport
        transport = cloudTransport;
    }
    else {
        const options = { logName: 'winston_log', ...optionsOrTransport };
        transport = new index_1.LoggingWinston(options);
        logger.add(transport);
    }
    const auth = (transport.common.redirectToStdout
        ? transport.common.cloudLog
        : transport.common.cloudLog).logging.auth;
    const [env, projectId] = await Promise.all([
        auth.getEnv(),
        auth.getProjectId(),
    ]);
    // Unless we are running on Google App Engine or Cloud Functions, generate a
    // parent request log entry that all the request specific logs ("app logs")
    // will nest under. GAE and GCF generate the parent request logs
    // automatically.
    // Cloud Run also generates the parent request log automatically, but skipping
    // the parent request entry has to be explicity enabled until the next major
    // release in which we can change the default behavior.
    let emitRequestLogEntry;
    if (env !== google_auth_library_1.GCPEnv.APP_ENGINE &&
        env !== google_auth_library_1.GCPEnv.CLOUD_FUNCTIONS &&
        (env !== google_auth_library_1.GCPEnv.CLOUD_RUN || !skipParentEntryForCloudRun)) {
        const requestLogName = logging_1.Log.formatName_(projectId, `${transport.common.logName}${exports.REQUEST_LOG_SUFFIX}`);
        emitRequestLogEntry = (httpRequest, trace, span, sampled) => {
            logger.info({
                // The request logs must have a log name distinct from the app logs
                // for log correlation to work.
                logName: requestLogName,
                [common_1.LOGGING_TRACE_KEY]: trace,
                [common_1.LOGGING_SPAN_KEY]: span,
                [common_1.LOGGING_SAMPLED_KEY]: sampled,
                httpRequest,
                message: httpRequest.requestUrl || 'http request',
            });
        };
    }
    return logging_1.middleware.express.makeMiddleware(projectId, (trace, span, sampled) => (0, make_child_logger_1.makeChildLogger)(logger, trace, span, sampled), emitRequestLogEntry);
}
//# sourceMappingURL=express.js.map